
Description: Kernel entry point and main initialization
Language: c
Lines: 724

/*
 * SMACKTM Kernel Main
 * Entry point and system initialization
 */

#include "kernel.h"
#include "mm/memory.h"
#include "drivers/vga.h"
#include "drivers/keyboard.h"
#include "fs/vfs.h"
#include "proc/scheduler.h"
#include "net/network.h"

// Kernel version info
#define KERNEL_VERSION_MAJOR 1
#define KERNEL_VERSION_MINOR 0
#define KERNEL_VERSION_PATCH 0

// System info structure
struct system_info {
    uint32_t total_memory;
    uint32_t available_memory;
    uint32_t kernel_size;
    char cpu_vendor[13];
    uint32_t cpu_features;
};

static struct system_info sysinfo;

/*
 * Kernel main entry point
 * Called from boot loader after system initialization
 */
void kernel_main(void) {
    // Initialize VGA display first
    vga_init();
    vga_clear();
    
    // Print kernel banner
    print_banner();
    
    kprintf("SMACKTM Kernel v%d.%d.%d starting...\n", 
            KERNEL_VERSION_MAJOR, KERNEL_VERSION_MINOR, KERNEL_VERSION_PATCH);
    
    // Detect system hardware
    kprintf("Detecting hardware...\n");
    detect_hardware();
    
    // Initialize memory management
    kprintf("Initializing memory management...\n");
    if (mm_init() != 0) {
        kernel_panic("Failed to initialize memory management");
    }
    
    // Initialize interrupt handling
    kprintf("Setting up interrupt handlers...\n");
    idt_init();
    irq_init();
    
    // Enable interrupts
    asm volatile("sti");
    
    // Initialize subsystems
    init_subsystems();
    
    // Start scheduler
    kprintf("Starting task scheduler...\n");
    scheduler_init();
    
    // Create init process
    create_init_process();
    
    kprintf("Kernel initialization complete.\n");
    kprintf("System ready. Entering idle loop...\n");
    
    // Kernel idle loop
    while (1) {
        asm volatile("hlt");  // Halt until interrupt
    }
}

static void detect_hardware(void) {
    // Detect CPU and memory
    uint32_t eax, ebx, ecx, edx;
    
    // Get CPU vendor string
    asm volatile("cpuid" 
                 : "=a"(eax), "=b"(ebx), "=c"(ecx), "=d"(edx)
                 : "a"(0));
    
    sysinfo.total_memory = detect_memory();
    kprintf("Total RAM: %d MB\n", sysinfo.total_memory / 1024 / 1024);
}

static void init_subsystems(void) {
    kprintf("Initializing keyboard driver...\n");
    keyboard_init();
    
    kprintf("Initializing file system...\n");
    vfs_init();
    
    kprintf("Initializing network stack...\n");
    network_init();
}

static void print_banner(void) {
    vga_set_color(VGA_COLOR_LIGHT_CYAN, VGA_COLOR_BLACK);
    kprintf("====================================\n");
    kprintf("    SMACKTM Operating System\n");
    kprintf("    Advanced Kernel Architecture\n");
    kprintf("====================================\n");
    vga_set_color(VGA_COLOR_LIGHT_GREY, VGA_COLOR_BLACK);
}

static void kernel_panic(const char* message) {
    cli();  // Disable interrupts
    
    vga_set_color(VGA_COLOR_WHITE, VGA_COLOR_RED);
    vga_clear();
    
    kprintf("\n\n*** KERNEL PANIC ***\n");
    kprintf("System halted due to critical error:\n");
    kprintf("%s\n", message);
    kprintf("\nSystem must be restarted.\n");
    
    // Halt the system
    while (1) {
        asm volatile("hlt");
    }
}

================================================================================

