
Description: Main kernel header with core definitions and structures
Language: header
Lines: 289

/*
 * SMACKTM Kernel Header
 * Core definitions and function prototypes
 */

#ifndef KERNEL_H
#define KERNEL_H

#include <stdint.h>
#include <stddef.h>
#include <stdbool.h>

// Kernel constants
#define KERNEL_STACK_SIZE    0x4000
#define MAX_PROCESSES        256
#define MAX_OPEN_FILES       1024
#define PAGE_SIZE            4096

// Memory layout constants
#define KERNEL_BASE          0x100000
#define USER_BASE            0x400000
#define KERNEL_HEAP_START    0x200000
#define KERNEL_HEAP_SIZE     0x200000

// System call numbers
#define SYS_EXIT             1
#define SYS_FORK             2
#define SYS_READ             3
#define SYS_WRITE            4
#define SYS_OPEN             5
#define SYS_CLOSE            6

// Process states
typedef enum {
    PROC_RUNNING,
    PROC_READY,
    PROC_BLOCKED,
    PROC_ZOMBIE,
    PROC_DEAD
} proc_state_t;

// Process structure
struct process {
    uint32_t pid;
    uint32_t ppid;
    proc_state_t state;
    uint32_t esp;
    uint32_t ebp;
    uint32_t eip;
    uint32_t cr3;  // Page directory
    
    // Memory regions
    uint32_t code_start;
    uint32_t code_end;
    uint32_t data_start;
    uint32_t data_end;
    uint32_t heap_start;
    uint32_t heap_end;
    uint32_t stack_start;
    uint32_t stack_end;
    
    char name[32];
};

// Function prototypes
void kernel_main(void);
void kernel_panic(const char* message);

// Memory management
int mm_init(void);
void* kmalloc(size_t size);
void kfree(void* ptr);
uint32_t detect_memory(void);

// Process management
struct process* create_process(const char* name);
void destroy_process(struct process* proc);
void schedule(void);
void yield(void);

// Interrupt handling
void idt_init(void);
void irq_init(void);

// Utility functions
void kprintf(const char* format, ...);
void cli(void);
void sti(void);

#endif // KERNEL_H

================================================================================


### MM MODULE ###

