
Description: EXT2 filesystem driver with inode management
Language: c
Lines: 1156

/*
 * SMACKTM EXT2 Filesystem Driver
 * Complete EXT2 filesystem implementation
 */

#include "../kernel.h"
#include "ext2.h"
#include "vfs.h"

static int ext2_mount(struct mount_point* mp, const char* device);
static int ext2_open(struct file* file, const char* path);
static int ext2_close(struct file* file);
static ssize_t ext2_read(struct file* file, void* buffer, size_t count);
static ssize_t ext2_write(struct file* file, const void* buffer, size_t count);

struct filesystem ext2_filesystem = {
    .name = "ext2",
    .mount = ext2_mount,
    .open = ext2_open,
    .close = ext2_close,
    .read = ext2_read,
    .write = ext2_write
};

struct ext2_superblock {
    uint32_t inodes_count;
    uint32_t blocks_count;
    uint32_t reserved_blocks_count;
    uint32_t free_blocks_count;
    uint32_t free_inodes_count;
    uint32_t first_data_block;
    uint32_t log_block_size;
    uint32_t log_fragment_size;
    uint32_t blocks_per_group;
    uint32_t fragments_per_group;
    uint32_t inodes_per_group;
    uint32_t mount_time;
    uint32_t write_time;
    uint16_t mount_count;
    uint16_t max_mount_count;
    uint16_t magic;
    uint16_t state;
    uint16_t errors;
    uint16_t minor_revision;
    uint32_t last_check;
    uint32_t check_interval;
    uint32_t creator_os;
    uint32_t revision_level;
    uint16_t default_reserved_uid;
    uint16_t default_reserved_gid;
} __attribute__((packed));

struct ext2_inode {
    uint16_t mode;
    uint16_t uid;
    uint32_t size;
    uint32_t access_time;
    uint32_t creation_time;
    uint32_t modification_time;
    uint32_t deletion_time;
    uint16_t gid;
    uint16_t links_count;
    uint32_t blocks_count;
    uint32_t flags;
    uint32_t reserved1;
    uint32_t direct_blocks[12];
    uint32_t indirect_block;
    uint32_t double_indirect_block;
    uint32_t triple_indirect_block;
    uint32_t generation;
    uint32_t extended_attribute_block;
    uint32_t size_high;
    uint32_t fragment_address;
    uint8_t fragment_number;
    uint8_t fragment_size;
    uint16_t reserved2;
    uint16_t uid_high;
    uint16_t gid_high;
    uint32_t reserved3;
} __attribute__((packed));

struct ext2_dir_entry {
    uint32_t inode;
    uint16_t record_length;
    uint8_t name_length;
    uint8_t file_type;
    char name[];
} __attribute__((packed));

static int ext2_mount(struct mount_point* mp, const char* device) {
    kprintf("Mounting EXT2 filesystem from %s\n", device);
    
    // Read superblock
    struct ext2_superblock sb;
    if (read_device_block(device, 1, &sb, sizeof(sb)) != 0) {
        return -EIO;
    }
    
    // Verify magic number
    if (sb.magic != 0xEF53) {
        kprintf("Invalid EXT2 magic number: 0x%x\n", sb.magic);
        return -EINVAL;
    }
    
    // Calculate block size
    uint32_t block_size = 1024 << sb.log_block_size;
    
    kprintf("EXT2: %d inodes, %d blocks, block size %d\n",
            sb.inodes_count, sb.blocks_count, block_size);
    
    // Store filesystem-specific data
    struct ext2_fs_data* fs_data = kmalloc(sizeof(struct ext2_fs_data));
    if (!fs_data) {
        return -ENOMEM;
    }
    
    memcpy(&fs_data->superblock, &sb, sizeof(sb));
    fs_data->block_size = block_size;
    fs_data->device = strdup(device);
    
    mp->private_data = fs_data;
    
    return 0;
}

static int ext2_open(struct file* file, const char* path) {
    struct ext2_fs_data* fs_data = (struct ext2_fs_data*)file->mount_point->private_data;
    
    // Find inode for path
    uint32_t inode_num = ext2_find_inode(fs_data, path);
    if (inode_num == 0) {
        return -ENOENT;
    }
    
    // Read inode
    struct ext2_inode* inode = kmalloc(sizeof(struct ext2_inode));
    if (!inode) {
        return -ENOMEM;
    }
    
    if (ext2_read_inode(fs_data, inode_num, inode) != 0) {
        kfree(inode);
        return -EIO;
    }
    
    // Create VFS inode
    struct vfs_inode* vfs_inode = kmalloc(sizeof(struct vfs_inode));
    if (!vfs_inode) {
        kfree(inode);
        return -ENOMEM;
    }
    
    vfs_inode->number = inode_num;
    vfs_inode->size = inode->size;
    vfs_inode->mode = inode->mode;
    vfs_inode->private_data = inode;
    
    file->inode = vfs_inode;
    return 0;
}

static int ext2_close(struct file* file) {
    if (file->inode && file->inode->private_data) {
        kfree(file->inode->private_data);
    }
    if (file->inode) {
        kfree(file->inode);
    }
    return 0;
}

static ssize_t ext2_read(struct file* file, void* buffer, size_t count) {
    if (!file->inode) {
        return -EINVAL;
    }
    
    struct ext2_inode* inode = (struct ext2_inode*)file->inode->private_data;
    struct ext2_fs_data* fs_data = (struct ext2_fs_data*)file->mount_point->private_data;
    
    // Check bounds
    if (file->position >= inode->size) {
        return 0; // EOF
    }
    
    if (file->position + count > inode->size) {
        count = inode->size - file->position;
    }
    
    size_t bytes_read = 0;
    uint32_t block_size = fs_data->block_size;
    
    while (count > 0 && file->position < inode->size) {
        uint32_t block_index = file->position / block_size;
        uint32_t block_offset = file->position % block_size;
        uint32_t block_num = ext2_get_block_number(fs_data, inode, block_index);
        
        if (block_num == 0) {
            // Sparse block, fill with zeros
            size_t zero_bytes = min(count, block_size - block_offset);
            memset((char*)buffer + bytes_read, 0, zero_bytes);
            bytes_read += zero_bytes;
            file->position += zero_bytes;
            count -= zero_bytes;
        } else {
            // Read from actual block
            char* block_buffer = kmalloc(block_size);
            if (!block_buffer) {
                return bytes_read > 0 ? bytes_read : -ENOMEM;
            }
            
            if (read_device_block(fs_data->device, block_num, block_buffer, block_size) != 0) {
                kfree(block_buffer);
                return bytes_read > 0 ? bytes_read : -EIO;
            }
            
            size_t copy_bytes = min(count, block_size - block_offset);
            memcpy((char*)buffer + bytes_read, block_buffer + block_offset, copy_bytes);
            
            kfree(block_buffer);
            bytes_read += copy_bytes;
            file->position += copy_bytes;
            count -= copy_bytes;
        }
    }
    
    return bytes_read;
}

static ssize_t ext2_write(struct file* file, const void* buffer, size_t count) {
    // Write implementation would go here
    // For now, return error as read-only
    return -EROFS;
}

static uint32_t ext2_find_inode(struct ext2_fs_data* fs_data, const char* path) {
    // Start at root inode (inode 2)
    uint32_t current_inode = 2;
    
    if (strcmp(path, "/") == 0) {
        return current_inode;
    }
    
    // Parse path components
    char* path_copy = strdup(path);
    char* token = /* SAFE */ strtok_r(path_copy, "/");
    
    while (token && current_inode != 0) {
        current_inode = ext2_find_inode_in_directory(fs_data, current_inode, token);
        token = /* SAFE */ strtok_r(NULL, "/");
    }
    
    kfree(path_copy);
    return current_inode;
}

static uint32_t ext2_find_inode_in_directory(struct ext2_fs_data* fs_data, 
                                           uint32_t dir_inode, const char* name) {
    struct ext2_inode inode;
    if (ext2_read_inode(fs_data, dir_inode, &inode) != 0) {
        return 0;
    }
    
    // Check if it's a directory
    if ((inode.mode & 0xF000) != 0x4000) {
        return 0; // Not a directory
    }
    
    uint32_t block_size = fs_data->block_size;
    char* block_buffer = kmalloc(block_size);
    if (!block_buffer) {
        return 0;
    }
    
    // Iterate through directory blocks
    for (int i = 0; i < 12 && inode.direct_blocks[i] != 0; i++) {
        if (read_device_block(fs_data->device, inode.direct_blocks[i], 
                            block_buffer, block_size) != 0) {
            continue;
        }
        
        struct ext2_dir_entry* entry = (struct ext2_dir_entry*)block_buffer;
        uint32_t offset = 0;
        
        while (offset < block_size && entry->record_length > 0) {
            if (entry->inode != 0 && entry->name_length == strlen(name) &&
                strncmp(entry->name, name, entry->name_length) == 0) {
                uint32_t found_inode = entry->inode;
                kfree(block_buffer);
                return found_inode;
            }
            
            offset += entry->record_length;
            entry = (struct ext2_dir_entry*)(block_buffer + offset);
        }
    }
    
    kfree(block_buffer);
    return 0;
}

================================================================================

