
Description: PCI bus enumeration and device discovery
Language: c
Lines: 589

/*
 * SMACKTM PCI Driver
 * PCI bus enumeration and device management
 */

#include "../kernel.h"
#include "pci.h"

#define PCI_CONFIG_ADDRESS  0xCF8
#define PCI_CONFIG_DATA     0xCFC

static struct pci_device pci_devices[256];
static int device_count = 0;

void pci_init(void) {
    kprintf("Scanning PCI bus...\n");
    
    device_count = 0;
    
    // Scan all buses, devices, and functions
    for (int bus = 0; bus < 256; bus++) {
        for (int device = 0; device < 32; device++) {
            for (int function = 0; function < 8; function++) {
                uint16_t vendor_id = pci_read_config_word(bus, device, function, 0);
                
                if (vendor_id != 0xFFFF) {
                    scan_pci_device(bus, device, function);
                }
                
                // If not multi-function device, skip other functions
                if (function == 0) {
                    uint8_t header_type = pci_read_config_byte(bus, device, function, 0x0E);
                    if ((header_type & 0x80) == 0) {
                        break;
                    }
                }
            }
        }
    }
    
    kprintf("Found %d PCI devices\n", device_count);
}

static void scan_pci_device(int bus, int device, int function) {
    if (device_count >= 256) return;
    
    struct pci_device* dev = &pci_devices[device_count];
    
    dev->bus = bus;
    dev->device = device;
    dev->function = function;
    dev->vendor_id = pci_read_config_word(bus, device, function, 0x00);
    dev->device_id = pci_read_config_word(bus, device, function, 0x02);
    dev->class_code = pci_read_config_byte(bus, device, function, 0x0B);
    dev->subclass = pci_read_config_byte(bus, device, function, 0x0A);
    dev->prog_if = pci_read_config_byte(bus, device, function, 0x09);
    dev->revision = pci_read_config_byte(bus, device, function, 0x08);
    
    // Read BAR registers
    for (int i = 0; i < 6; i++) {
        dev->bar[i] = pci_read_config_dword(bus, device, function, 0x10 + i * 4);
    }
    
    kprintf("PCI %02x:%02x.%x - %04x:%04x (Class %02x:%02x)\n",
            bus, device, function, dev->vendor_id, dev->device_id,
            dev->class_code, dev->subclass);
    
    device_count++;
}

uint32_t pci_read_config_dword(int bus, int device, int function, int offset) {
    uint32_t address = (1 << 31) | (bus << 16) | (device << 11) | 
                       (function << 8) | (offset & 0xFC);
    
    outl(PCI_CONFIG_ADDRESS, address);
    return inl(PCI_CONFIG_DATA);
}

uint16_t pci_read_config_word(int bus, int device, int function, int offset) {
    uint32_t dword = pci_read_config_dword(bus, device, function, offset);
    return (uint16_t)(dword >> ((offset & 2) * 8));
}

uint8_t pci_read_config_byte(int bus, int device, int function, int offset) {
    uint32_t dword = pci_read_config_dword(bus, device, function, offset);
    return (uint8_t)(dword >> ((offset & 3) * 8));
}

struct pci_device* find_pci_device(uint16_t vendor_id, uint16_t device_id) {
    for (int i = 0; i < device_count; i++) {
        if (pci_devices[i].vendor_id == vendor_id && 
            pci_devices[i].device_id == device_id) {
            return &pci_devices[i];
        }
    }
    return NULL;
}

================================================================================


### FS MODULE ###

