
Description: PS/2 keyboard driver with scancode translation
Language: c
Lines: 632

/*
 * SMACKTM Keyboard Driver
 * PS/2 keyboard driver with scancode handling
 */

#include "../kernel.h"
#include "keyboard.h"

// Keyboard ports
#define KEYBOARD_DATA_PORT   0x60
#define KEYBOARD_STATUS_PORT 0x64
#define KEYBOARD_COMMAND_PORT 0x64

// Special keys
#define KEY_CTRL    0x1D
#define KEY_ALT     0x38
#define KEY_SHIFT_L 0x2A
#define KEY_SHIFT_R 0x36
#define KEY_CAPS    0x3A

// Keyboard state
static bool shift_pressed = false;
static bool ctrl_pressed = false;
static bool alt_pressed = false;
static bool caps_lock = false;

// Input buffer
static char input_buffer[256];
static size_t buffer_head = 0;
static size_t buffer_tail = 0;
static size_t buffer_count = 0;

/*
 * Initialize keyboard driver
 */
void keyboard_init(void) {
    kprintf("Initializing keyboard driver...\n");
    
    // Clear input buffer
    buffer_head = 0;
    buffer_tail = 0;
    buffer_count = 0;
    
    // Reset keyboard state
    shift_pressed = false;
    ctrl_pressed = false;
    alt_pressed = false;
    caps_lock = false;
    
    // Register keyboard interrupt handler
    register_interrupt_handler(33, keyboard_interrupt_handler);
    
    // Enable keyboard
    keyboard_enable();
    
    kprintf("Keyboard driver initialized\n");
}

/*
 * Keyboard interrupt handler
 */
void keyboard_interrupt_handler(void) {
    uint8_t scancode = inb(KEYBOARD_DATA_PORT);
    
    // Handle key release (high bit set)
    if (scancode & 0x80) {
        scancode &= 0x7F;  // Remove release bit
        
        switch (scancode) {
            case KEY_SHIFT_L:
            case KEY_SHIFT_R:
                shift_pressed = false;
                break;
            case KEY_CTRL:
                ctrl_pressed = false;
                break;
            case KEY_ALT:
                alt_pressed = false;
                break;
        }
        return;
    }
    
    // Handle special keys
    switch (scancode) {
        case KEY_SHIFT_L:
        case KEY_SHIFT_R:
            shift_pressed = true;
            return;
        case KEY_CTRL:
            ctrl_pressed = true;
            return;
        case KEY_ALT:
            alt_pressed = true;
            return;
        case KEY_CAPS:
            caps_lock = !caps_lock;
            return;
    }
}

/*
 * Read character from keyboard buffer
 */
int keyboard_getchar(void) {
    if (buffer_count == 0) {
        return -1;  // No character available
    }
    
    char c = input_buffer[buffer_head];
    buffer_head = (buffer_head + 1) % sizeof(input_buffer);
    buffer_count--;
    
    return (int)c;
}

/*
 * Check if character is available
 */
bool keyboard_char_available(void) {
    return buffer_count > 0;
}

================================================================================

