extends Node2D

@export_category("Colors")
@export var GrassColor : ColorGroup 
@export var BorderColor : ColorGroup
@export var RoadColor : ColorGroup 
@export var StripesColor : ColorGroup 

@export_category("Road")
@export var RoadData : Array[RoadDataItem] = []

@export_category("Sizes")
@export var TotalWidth : int = 1152
@export var TotalHeight : int = 648
@export var RoadWidth : int = 2000 

var segments : float = 200
var cam : float = 0.85
var lines : Array[LineData] = []
var currentPosition : float = 0
var totalLines : int
var skylineX : float = 0.0
var skylineY : float = 0
var roadLenght : float = 2000
var direction : float = 0

@onready var skyline: Sprite2D = $Skyline

func _ready():
	var lastElevation :  float = 0
	for i in range(roadLenght):
		var line = LineData.new()
		line.Z = i * segments + 0.00000001
		var dataRules = RoadData.filter(func(d : RoadDataItem) : return i >= d.Min and i <= d.Max)
		for dataRule in dataRules:
			if dataRule.Type == 0:
				line.Curvature = dataRule.Value
			else:
				line.Y = sin(i / dataRule.Value) * 1500
				lastElevation = line.Y
		if len(dataRules.filter(func(d : RoadDataItem) : return d.Type == 1)) == 0 and lastElevation != 0:
			line.Y = lerpf(line.Y, 0, 0.9)
			lastElevation = line.Y
		lines.append(line)
	totalLines = lines.size()

func _process(_delta: float) -> void:
	direction = Input.get_axis("ui_down", "ui_up")
	currentPosition += segments * direction
	queue_redraw()

func _draw() -> void:
	if currentPosition >= totalLines * segments:
		currentPosition -= totalLines * segments
		
	if currentPosition < 0:
		currentPosition += totalLines * segments
			
	var numPos = 0
	var startPoint = currentPosition / segments
	var camH = 1500 + lines[startPoint].Y
	var cutoff = TotalHeight
	var baseX = 0
	var topX = 0
	
	skylineX += lines[startPoint].Curvature * 2.0 * direction
	skylineY = -lines[startPoint].Y * 0.005
	skyline.set_region_rect(Rect2(skylineX, skylineY, 1920, 640))

	for index in range(startPoint, startPoint + segments):
		if index >= totalLines:
			numPos = totalLines * segments
		else:
			numPos = 0
			
		var topLine = Project(lines[fmod(index, totalLines)], -baseX, camH, currentPosition - numPos)
		var baseLine = lines[fmod(index - 1, totalLines)]
		baseX += topX
		topX += topLine.Curvature
		
		if topLine.CamY >= cutoff:
			continue
			
		cutoff = topLine.CamY
		
		var grassColor = GrassColor.Dark if fmod((index / 8), 2) else GrassColor.Light
		var borderColor = BorderColor.Dark if fmod((index / 4), 2) else BorderColor.Light
		var roadColor = RoadColor.Dark if fmod((index / 4), 2) else RoadColor.Light
		var stripesColor = StripesColor.Dark if fmod((index / 8), 2) else StripesColor.Light
		
		DrawSegment(grassColor, 0, baseLine.CamY, TotalWidth, 0, topLine.CamY, TotalWidth)
		DrawSegment(borderColor, baseLine.CamX, baseLine.CamY, baseLine.CamW * 1.2, topLine.CamX, topLine.CamY, topLine.CamW * 1.2)
		DrawSegment(roadColor, baseLine.CamX, baseLine.CamY, baseLine.CamW, topLine.CamX, topLine.CamY, topLine.CamW)
		DrawSegment(stripesColor, baseLine.CamX, baseLine.CamY, baseLine.CamW * 0.01, topLine.CamX, topLine.CamY, topLine.CamW * 0.01)

func DrawSegment(color : Color, x1 : float, y1 : float, w1 : float, x2 : float, y2 : float, w2 : float):
	var point = [
		Vector2(int(x1 - w1), int(y1)), #A 
		Vector2(int(x2 - w2), int(y2)), #C
		Vector2(int(x2 + w2), int(y2)), #D
		Vector2(int(x1 + w1), int(y1))  #B
		]
	draw_primitive(PackedVector2Array(point), PackedColorArray([color]),PackedVector2Array([]))

func Project(line : LineData, cam_x : float, cam_y : float, cam_z : float):
	line.Scale = cam / (line.Z - cam_z)
	line.CamX = (1 + line.Scale * (line.X - cam_x)) * TotalWidth / 2
	line.CamY = (1 - line.Scale * (line.Y - cam_y)) * TotalHeight / 2
	line.CamW = line.Scale * RoadWidth * (TotalWidth / 2.0)
	return line
